% plotData.m
%
% Matlab script for plotting experimental data
%
% Save this script in the same directory as the Matlab data files,
% dataMatlab1.m, dataMatlab2.m, etc. or change the dataPath variable to
% the directory where you have saved those data files.
%
clc; clear; close all;

% path to directory where the data files are located
dataPath = pwd;

% number of perturbed cases
Ncases = 45;

% load in the data file for each case
disp('loading data files from directory:')
disp(dataPath); disp(' ')

X{Ncases+1} = [];
Y{Ncases+1} = [];
U{Ncases+1} = [];
V{Ncases+1} = [];
valid{Ncases+1} = [];
for j = 1:Ncases
    dataFile = [dataPath,'/dataMatlab_case',num2str(j),'.mat'];
    data = load(dataFile,'X','Y','U','V','valid');
    X{j} = data.X;
    Y{j} = data.Y;
    U{j} = data.U;
    V{j} = data.V;
    valid{j} = data.valid;
    clear data
end

% load in unperturbed case data
dataFile = [dataPath,'/dataMatlab_unperturbed.mat'];
data = load(dataFile,'X','Y','U','V','valid');
X{Ncases+1} = data.X;
Y{Ncases+1} = data.Y;
U{Ncases+1} = data.U;
V{Ncases+1} = data.V;
valid{Ncases+1} = data.valid;
clear data

% load info about each perturbed case and the shock crossing points
load([dataPath,'/dataSummary.mat'],...
    'x_scp_unperturbed','y_scp_unperturbed',...
    'Delta_x_scp_vector','Delta_y_scp_vector',...
    'h_bump_vector','x_bump_vector')

% display info for unperturbed case
disp('shock crossing point location for unperturbed case:')
disp(['x_scp_unperturbed = ',num2str(x_scp_unperturbed, '%1.2f'),'mm'])
disp(['y_scp_unperturbed = ',num2str(y_scp_unperturbed, '%1.2f'),'mm'])
disp(' ')

% plot data for each case
for j = 1:Ncases + 1
    
    % crop out any invalid data regions (NaNs display blank)
    U{j}(~valid{j}) = NaN;
    V{j}(~valid{j}) = NaN;
    
    % plot velocity fields
    figure(j);
    set(gcf,'units','pixels','position',[100 100 1000 400])
    set(gcf,'color','w')
    subplot(1,2,1)
    surf(X{j},Y{j},U{j},'linestyle','none'); view(2);
    set(gca,'CLim',[0 550])
    if j <= Ncases
        title(['$U$ [m/s]: perturbed case ',num2str(j)],...
            'interpreter','latex','FontSize',13)
    else
        title('$U$ [m/s]: unperturbed case',...
            'interpreter','latex','FontSize',13)
    end
    subplot(1,2,2);
    surf(X{j},Y{j},V{j},'linestyle','none'); view(2);
    set(gca,'CLim',[-50 50])
    if j <= Ncases
        title(['$V$ [m/s]: perturbed case ',num2str(j)],...
            'interpreter','latex','FontSize',13)
    else
        title('$V$ [m/s]: unperturbed case',...
            'interpreter','latex','FontSize',13)
    end
    
    % format the plot nicely
    for subfigNum = 1:2
        subplot(1,2,subfigNum)
        set(gca,'FontSize',14)
        colorbar
        xlim([-10 12])
        ylim([0 16])
        xlabel('$x - x_{scp,unperturbed}$ [mm]',...
            'interpreter','latex')
        ylabel('$y$ [mm]','interpreter','latex')
    end
    
    % display info about each case
    if j <= Ncases
        xscpString = num2str(Delta_x_scp_vector(j),'%1.2f');
        yscpString = num2str(Delta_y_scp_vector(j),'%1.2f');
        disp([num2str(j),': x_bump = ',num2str(x_bump_vector(j)),...
            'mm, h_bump = ',num2str(h_bump_vector(j)),...
            'mm, Delta x_scp = ',xscpString,'mm',...
            'mm, Delta y_scp = ',yscpString,'mm'])
    end
end

% plot shock crossing point data for all cases
figure(100);
colors = {0.65*[1 1 1], 0.53*[1 1 1], 0.42*[1 1 1], 0.3*[1 1 1], [0 0 0]};
set(gcf,'units','pixels','position',[100 100 700 500])
set(gcf,'color','w')
set(gca,'FontSize',13)
plot(x_bump_vector(1:5:45),Delta_x_scp_vector(1:5:45),'x--',...
    'Color',colors{1},'linewidth',2); hold on
plot(x_bump_vector(2:5:45),Delta_x_scp_vector(2:5:45),'o--',...
    'Color',colors{2},'linewidth',2); hold on
plot(x_bump_vector(3:5:45),Delta_x_scp_vector(3:5:45),'s--',...
    'Color',colors{3},'linewidth',2); hold on
plot(x_bump_vector(4:5:45),Delta_x_scp_vector(4:5:45),'v--',...
    'Color',colors{4},'linewidth',2); hold on
plot(x_bump_vector(5:5:45),Delta_x_scp_vector(5:5:45),'+--',...
    'Color',colors{5},'linewidth',2); hold on
errorbar(x_bump_vector(40),Delta_x_scp_vector(40),0.25,'k-','linewidth',2)
hleg = legend('$h_{bump} = 0.11$mm','$h_{bump} = 0.28$mm',...
    '$h_{bump} = 0.48$mm','$h_{bump} = 0.66$mm','$h_{bump} = 0.89$mm');
set(hleg','Location','SouthWest','interpreter','latex','FontSize',12)
xlabel('$x_{bump}$ [mm]','interpreter','latex','FontSize',13)
ylabel('$\Delta x_{scp}$ [mm]','interpreter','latex','FontSize',13)
xlim([-80 -40])
ylim([-5 1])
grid on

