% This script plots the PIV datasets for the SBLI flow generated by a 5mm
% tall (hramp/delta0 = 0.93) 20 degree compression ramp in a low aspect
% ratio duct.  See Laura Campo's PhD thesis ("Effects of Shock Strength,
% Confinement, and Geometric Perturbations on Shock Boundary Layer 
% Interactions" Stanford University, 2014) for details of the experiments.

clear all; close all; clc;
if ispc; opengl software; end
set(0, 'DefaultAxesFontSize', 13);
colormap_file = 'customColormaps.mat';
load(colormap_file)

disp('this script plots PIV datasets for the SBLI flow')
disp('generated by a 5mm tall 20 degree compression ramp'); disp(' ')
disp('data from 4 different planes are available')
disp('options are:')
disp('z = 21mm (z/delta0 = 3.89)')
disp('z = 8mm (z/delta0 = 1.48)')
disp('z = 5.5mm (z/delta0 = 1.02)')
disp('z = 4mm (z/delta0 = 0.74)')
disp('type the z-coordinate of the data plane')
dataPlane = input('that you want to plot (21, 8, 5.5, or 4) ');
dataFile = ['data_h=5mm_ramp_Z=',num2str(dataPlane),'mm_plane.mat'];
disp(' '); disp(['using data file: ',dataFile])
load(dataFile)

% define the top & bottom wall geometries
alpha = 20;
hramp = 5;
lramp = hramp/tand(alpha);
h_topwall = 45;
x_wall = linspace(-40,90,1000);
y_wall_bot = zeros(size(x_wall));
upstream = x_wall <= 0;
along_wedge = x_wall > 0 & x_wall <= lramp;
downstream = x_wall > lramp;
y_wall_top = h_topwall.*upstream + (h_topwall - hramp).*downstream + ...
    (h_topwall - x_wall*tand(alpha)).*along_wedge;
    
%% plot all of the data for the selected Z plane
for i = 1:numel(X)
    
    % plot U
    figure(1)
    h_U = pcolor(X{i},Y{i},U{i}); hold on
    set(h_U,'ZData',-1+0*U{i})
    shading interp
    
    % plot V
    figure(2)
    h_V = pcolor(X{i},Y{i},V{i}); hold on
    set(h_V,'ZData',-1+0*V{i})
    shading interp
    
    % plot u'/Uinf
    figure(3)
    h_uprime = pcolor(X{i},Y{i},uprime{i}); hold on
    set(h_uprime,'ZData',-1+0*uprime{i})
    shading interp
    
    % plot sqrt(v'v')/Uinf
    figure(4)
    h_vprime = pcolor(X{i},Y{i},vprime{i}); hold on
    set(h_vprime,'ZData',-1+0*vprime{i})
    shading interp
    
    % plot (u'v')/Uinf^2
    figure(5)
    h_uv = pcolor(X{i},Y{i},1000*uprimevprime{i}); hold on
    set(h_uv,'ZData',-1+0*uprimevprime{i})
    shading interp
end

%% format plots

% plotting ranges
y_range = [-0.2 45.2];
x_range = [-40 85];
v_range = [-100 100];
u_range = [-8 550];
uprime_range = [0 .3];
vprime_range = [0 .15];
uprimevprime_range = [-5 5];
 % axis label properties
xaxis_string = '$x$ [mm]';
yaxis_string = '$y$ [mm]';
tick_fsize = 20;
axis_fsize = 22;
xtitle = 25;                
xtitle2 = 55;              
ytitle = 45;
% colorbar properties
colorbar_fsize = 20;
colorbar_font = 'Arial';
cbar_position = [0.9 0.12 0.02 0.8];
axis_position = [0.07 0.12 0.8 0.8];
bkgd_color = 'w';

% U
figure(1)
text(xtitle,ytitle,'$U$ [m/s]','interpreter','latex','FontSize', axis_fsize)
set(gcf,'Colormap',cmap_u)
set(gca, 'CLim', u_range)
cbar_h = colorbar('location','manual','position',cbar_position);
set(cbar_h, ...
    'YTick'       , 0:50:550   , ...
    'FontSize'    , colorbar_fsize    , ...
    'FontName'    , colorbar_font);

% V
figure(2)
text(xtitle,ytitle,'$V$ [m/s]','interpreter','latex','FontSize', axis_fsize)
set(gcf,'Colormap',cmap_v)
set(gca, 'CLim', v_range)
cbar_h = colorbar('location','manual','position',cbar_position);
set(cbar_h, ...
    'YTick'       , -100:25:100   , ...
    'FontSize'    , colorbar_fsize    , ...
    'FontName'    , colorbar_font);   

% u'/Uinf
figure(3)
text(xtitle,ytitle,'$u''/U_{\infty}$','interpreter','latex','FontSize', axis_fsize)
set(gcf,'Colormap',cmap_uprime)
set(gca, 'CLim', uprime_range)
cbar_h = colorbar('location','manual','position',cbar_position);
set(cbar_h, ...
    'YTick'       , 0:0.05:0.3   , ...
    'FontSize'    , colorbar_fsize    , ...
    'FontName'    , colorbar_font);

% v'/Uinf
figure(4)
text(xtitle,ytitle,'$v''/U_{\infty}$','interpreter','latex','FontSize', axis_fsize)
set(gcf,'Colormap',cmap_uprime)
set(gca, 'CLim', vprime_range)
cbar_h = colorbar('location','manual','position',cbar_position);
set(cbar_h, ...
    'YTick'       , 0:0.05:0.15   , ...
    'FontSize'    , colorbar_fsize    , ...
    'FontName'    , colorbar_font);

% u'v'/Uinf^2
figure(5)
text(xtitle,ytitle,'$u''v''/U_{\infty}^2 \times 10^3$','interpreter','latex','FontSize', axis_fsize)
set(gcf,'Colormap',cmap_uv)
set(gca, 'CLim', uprimevprime_range)
cbar_h = colorbar('location','manual','position',cbar_position);
set(cbar_h, ...
    'YTick'       , -5:1:5   , ...
    'FontSize'    , colorbar_fsize    , ...
    'FontName'    , colorbar_font);

% all plots
for j = 1:5
    figure(j)
    plot(x_wall,y_wall_top,'k','linewidth',2);
    plot(x_wall,y_wall_bot,'k','linewidth',2)
    xlabel(xaxis_string,'FontSize',axis_fsize,'interpreter','latex')
    ylabel(yaxis_string,'FontSize',axis_fsize,'interpreter','latex')
    text(xtitle2,ytitle,['$z = $', num2str(dataPlane),'mm'],'interpreter','latex','FontSize', axis_fsize)
    set(gcf,'Color','w')
    set(gcf,'Units','pixels','Position',[100 200 1200 700])
    set(gca, ...
        'FontSize'    , tick_fsize, ...
        'XLim'        , x_range, ...
        'YLim'        , y_range, ...
        'Box'         , 'off', ...
        'TickDir'     , 'out', ...
        'TickLength'  , [.007 .007], ...
        'XMinorTick'  , 'off', ...
        'YMinorTick'  , 'off', ...
        'YGrid'       , 'off', ...
        'XGrid'       , 'off', ...
        'Position'    , axis_position)
end

 